from azure.digitaltwins.core import DigitalTwinsClient
from azure.identity import DeviceCodeCredential

class DigitalTwinsClientWrapper:
    def __init__(self, adt_instance_url: str):
        """
        Initializes the Digital Twins Client with Device Code Authentication.
        """
        try:
            # Device Code Auth (prints a code for login)
            self.credential = DeviceCodeCredential()
            self.client = DigitalTwinsClient(adt_instance_url, self.credential)
        except Exception as e:
            print(f"❌ Failed to initialize Digital Twins client: {e}")
            raise

    def send_data(self, twin_id: str, property_name: str, value):
        """
        Updates a property of a digital twin using JSON Patch 'add' operation.
        Example: send_data("Temps", "Tr", 45.5)
        """
        try:
            # Path must start with / for JSON Patch
            path = f"/{property_name}" if not property_name.startswith("/") else property_name
            patch = [
                {"op": "add", "path": path, "value": value}
            ]
            self.client.update_digital_twin(twin_id, patch)
            print(f"✅ Updated '{twin_id}.{property_name}' to {value}")
        except Exception as e:
            print(f"❌ Error sending data to twin {twin_id}.{property_name}: {e}")

    def get_data(self, twin_id: str, property_key: str):
        """
        Retrieves a property value from a digital twin.
        Example: get_data("Controls", "CoolantFlow")
        """
        try:
            twin = self.client.get_digital_twin(twin_id)
            value = twin.get(property_key, None)
            print(f"🔎 Retrieved {twin_id}.{property_key}: {value}")
            return value
        except Exception as e:
            print(f"❌ Error retrieving data from twin {twin_id}.{property_key}: {e}")
            return None

    # --- Specialized methods for the Reactor Graph ---

    def update_reactor_temperatures(self, tr=None, tj=None, tc=None):
        """Updates Tr, Tj, and Tc on the 'Temps' twin."""
        if tr is not None: self.send_data("Temps", "Tr", tr)
        if tj is not None: self.send_data("Temps", "Tj", tj)
        if tc is not None: self.send_data("Temps", "Tc", tc)

    def update_reactor_controls(self, coolant_flow=None, heater_current=None):
        """Updates CoolantFlow and HeaterCurrent on the 'Controls' twin."""
        if coolant_flow is not None: self.send_data("Controls", "CoolantFlow", coolant_flow)
        if heater_current is not None: self.send_data("Controls", "HeaterCurrent", heater_current)

    def update_reactor_state(self, predicted_tr=None, status=None):
        """Updates PredictedTr and Status on the 'States' twin."""
        if predicted_tr is not None: self.send_data("States", "PredictedTr", predicted_tr)
        if status is not None: self.send_data("States", "Status", status)

    def get_all_temperatures(self):
        """Retrieves all temperature values from the 'Temps' twin."""
        tr = self.get_data("Temps", "Tr")
        tj = self.get_data("Temps", "Tj")
        tc = self.get_data("Temps", "Tc")
        return {"Tr": tr, "Tj": tj, "Tc": tc}

    def get_all_controls(self):
        """Retrieves all control values from the 'Controls' twin."""
        coolant = self.get_data("Controls", "CoolantFlow")
        heater = self.get_data("Controls", "HeaterCurrent")
        return {"CoolantFlow": coolant, "HeaterCurrent": heater}
